/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import * as o from '../output/output_ast';
import { Identifiers as R3 } from './r3_identifiers';
import { jitOnlyGuardedExpression, refsToArray } from './util';
import { DefinitionMap } from './view/util';
/**
 * How the selector scope of an NgModule (its declarations, imports, and exports) should be emitted
 * as a part of the NgModule definition.
 */
export var R3SelectorScopeMode;
(function (R3SelectorScopeMode) {
    /**
     * Emit the declarations inline into the module definition.
     *
     * This option is useful in certain contexts where it's known that JIT support is required. The
     * tradeoff here is that this emit style prevents directives and pipes from being tree-shaken if
     * they are unused, but the NgModule is used.
     */
    R3SelectorScopeMode[R3SelectorScopeMode["Inline"] = 0] = "Inline";
    /**
     * Emit the declarations using a side effectful function call, `ɵɵsetNgModuleScope`, that is
     * guarded with the `ngJitMode` flag.
     *
     * This form of emit supports JIT and can be optimized away if the `ngJitMode` flag is set to
     * false, which allows unused directives and pipes to be tree-shaken.
     */
    R3SelectorScopeMode[R3SelectorScopeMode["SideEffect"] = 1] = "SideEffect";
    /**
     * Don't generate selector scopes at all.
     *
     * This is useful for contexts where JIT support is known to be unnecessary.
     */
    R3SelectorScopeMode[R3SelectorScopeMode["Omit"] = 2] = "Omit";
})(R3SelectorScopeMode || (R3SelectorScopeMode = {}));
/**
 * Construct an `R3NgModuleDef` for the given `R3NgModuleMetadata`.
 */
export function compileNgModule(meta) {
    const { adjacentType, internalType, bootstrap, declarations, imports, exports, schemas, containsForwardDecls, selectorScopeMode, id } = meta;
    const statements = [];
    const definitionMap = new DefinitionMap();
    definitionMap.set('type', internalType);
    if (bootstrap.length > 0) {
        definitionMap.set('bootstrap', refsToArray(bootstrap, containsForwardDecls));
    }
    if (selectorScopeMode === R3SelectorScopeMode.Inline) {
        // If requested to emit scope information inline, pass the `declarations`, `imports` and
        // `exports` to the `ɵɵdefineNgModule()` call directly.
        if (declarations.length > 0) {
            definitionMap.set('declarations', refsToArray(declarations, containsForwardDecls));
        }
        if (imports.length > 0) {
            definitionMap.set('imports', refsToArray(imports, containsForwardDecls));
        }
        if (exports.length > 0) {
            definitionMap.set('exports', refsToArray(exports, containsForwardDecls));
        }
    }
    else if (selectorScopeMode === R3SelectorScopeMode.SideEffect) {
        // In this mode, scope information is not passed into `ɵɵdefineNgModule` as it
        // would prevent tree-shaking of the declarations, imports and exports references. Instead, it's
        // patched onto the NgModule definition with a `ɵɵsetNgModuleScope` call that's guarded by the
        // `ngJitMode` flag.
        const setNgModuleScopeCall = generateSetNgModuleScopeCall(meta);
        if (setNgModuleScopeCall !== null) {
            statements.push(setNgModuleScopeCall);
        }
    }
    else {
        // Selector scope emit was not requested, so skip it.
    }
    if (schemas !== null && schemas.length > 0) {
        definitionMap.set('schemas', o.literalArr(schemas.map(ref => ref.value)));
    }
    if (id !== null) {
        definitionMap.set('id', id);
        // Generate a side-effectful call to register this NgModule by its id, as per the semantics of
        // NgModule ids.
        statements.push(o.importExpr(R3.registerNgModuleType).callFn([adjacentType, id]).toStmt());
    }
    const expression = o.importExpr(R3.defineNgModule).callFn([definitionMap.toLiteralMap()], undefined, true);
    const type = createNgModuleType(meta);
    return { expression, type, statements };
}
/**
 * This function is used in JIT mode to generate the call to `ɵɵdefineNgModule()` from a call to
 * `ɵɵngDeclareNgModule()`.
 */
export function compileNgModuleDeclarationExpression(meta) {
    const definitionMap = new DefinitionMap();
    definitionMap.set('type', new o.WrappedNodeExpr(meta.type));
    if (meta.bootstrap !== undefined) {
        definitionMap.set('bootstrap', new o.WrappedNodeExpr(meta.bootstrap));
    }
    if (meta.declarations !== undefined) {
        definitionMap.set('declarations', new o.WrappedNodeExpr(meta.declarations));
    }
    if (meta.imports !== undefined) {
        definitionMap.set('imports', new o.WrappedNodeExpr(meta.imports));
    }
    if (meta.exports !== undefined) {
        definitionMap.set('exports', new o.WrappedNodeExpr(meta.exports));
    }
    if (meta.schemas !== undefined) {
        definitionMap.set('schemas', new o.WrappedNodeExpr(meta.schemas));
    }
    if (meta.id !== undefined) {
        definitionMap.set('id', new o.WrappedNodeExpr(meta.id));
    }
    return o.importExpr(R3.defineNgModule).callFn([definitionMap.toLiteralMap()]);
}
export function createNgModuleType({ type: moduleType, declarations, imports, exports }) {
    return new o.ExpressionType(o.importExpr(R3.NgModuleDeclaration, [
        new o.ExpressionType(moduleType.type), tupleTypeOf(declarations), tupleTypeOf(imports),
        tupleTypeOf(exports)
    ]));
}
/**
 * Generates a function call to `ɵɵsetNgModuleScope` with all necessary information so that the
 * transitive module scope can be computed during runtime in JIT mode. This call is marked pure
 * such that the references to declarations, imports and exports may be elided causing these
 * symbols to become tree-shakeable.
 */
function generateSetNgModuleScopeCall(meta) {
    const { adjacentType: moduleType, declarations, imports, exports, containsForwardDecls } = meta;
    const scopeMap = new DefinitionMap();
    if (declarations.length > 0) {
        scopeMap.set('declarations', refsToArray(declarations, containsForwardDecls));
    }
    if (imports.length > 0) {
        scopeMap.set('imports', refsToArray(imports, containsForwardDecls));
    }
    if (exports.length > 0) {
        scopeMap.set('exports', refsToArray(exports, containsForwardDecls));
    }
    if (Object.keys(scopeMap.values).length === 0) {
        return null;
    }
    // setNgModuleScope(...)
    const fnCall = new o.InvokeFunctionExpr(
    /* fn */ o.importExpr(R3.setNgModuleScope), 
    /* args */ [moduleType, scopeMap.toLiteralMap()]);
    // (ngJitMode guard) && setNgModuleScope(...)
    const guardedCall = jitOnlyGuardedExpression(fnCall);
    // function() { (ngJitMode guard) && setNgModuleScope(...); }
    const iife = new o.FunctionExpr(
    /* params */ [], 
    /* statements */ [guardedCall.toStmt()]);
    // (function() { (ngJitMode guard) && setNgModuleScope(...); })()
    const iifeCall = new o.InvokeFunctionExpr(
    /* fn */ iife, 
    /* args */ []);
    return iifeCall.toStmt();
}
function tupleTypeOf(exp) {
    const types = exp.map(ref => o.typeofExpr(ref.type));
    return exp.length > 0 ? o.expressionType(o.literalArr(types)) : o.NONE_TYPE;
}
//# sourceMappingURL=data:application/json;base64,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